% This program replicates Figure 5 in "A Static Capital Buffer is 
% Hard to Beat" by Matthew Canzoneri, Behzad Diba, Luca Guerrieri 
% and Arsenii Mishin.
%
% The program produces Figure 5, corresponding to an increase in
% investment-specific technology, formatted as in the paper and another figure formatted for
% the slides.

clear all

setpath_windows

%global M_ oo_
global overwrite overwrite_param_names



nperiods = 60;    % periods for IRFs
maxiter = 20;


% select shock and size
irfshock = char('eps_a');
shockssequence = -1;


% parameter changes
gammap_increment = 0.000012;
overwrite_param_names = char('gammap_increment');
overwrite = [gammap_increment];


% choose model files - these are the files for the case with regime
% switching
modbuffer = 'model_safe_all_shocks';
modnobuffer = 'model_safe_opt_all_shocks';
mod11 = 'model_mixed_all_shocks';
mod10 = 'model_risky_all_shocks';
mod01 = 'model_spare_all_shocks';

% Under the Ramsey rule there is no switching
modopt = 'model_calibration';


% define the constraints

constraint1 = 'chi2_upper_bar+chi2_upper_bar_ss<0';    % If the Lagrange multiplier on the
% no-shorting constraints for risky
% loans is slack, it means that I want
% to make risky loans.
% Accordingly, go into 11 or 10

constraint_relax1 = '((l+l_ss)<0)|(l_upper_bar<0)'; % If it turns out that the quantity of
% risky loans is negative,  00
% notice that l is set to 0 in the 10
% file, therefore the first condition
% is irrelevant when transition back
% from 10, but it might be relevant
% when transition back from 11.


constraint2 = '((chi2_lower_bar<0)&(l_upper_bar>0))'; % If the Lagrange multiplier on the
% no-shorting constraints for risky
% loans  and on safe loans are both slack, it means that I want
% to make risky and safe loans. Go to
% 01 or 11, but since if this
% condition is met constraint1 will
% also be active, we can only go into
% 11. Hence, 01 is spare.

constraint_relax2 ='((l+l_ss)<0)|(l_upper_bar<0)';  % return to 10 or 00, but
% notice that
% constraint_relax1 is identical,
% therefore, if either
% condition is violated
% we will go back to 00


% end of inputs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


[zdatalinear, zdatapiecewise, zdatass, oobase_, Mbase_, violvecbool ] = ...
    solve_two_constraints_add_violvecbool(modnobuffer,mod10,mod01,mod11,...
    constraint1, constraint2,...
    constraint_relax1, constraint_relax2,...
    shockssequence,irfshock,nperiods,0,maxiter);

[zdatalinear3, zdatapiecewise3, zdatass3, oobase3_, Mbase3_, violvecbool3 ] = ...
    solve_two_constraints_add_violvecbool(modbuffer,mod10,mod01,mod11,...
    constraint1, constraint2,...
    constraint_relax1, constraint_relax2,...
    shockssequence,irfshock,nperiods,0,maxiter);

% repeat the simulation for the model with endogenous capital requirements

% set the static buffer to 0 when using the optimal rule
gammap_increment = 0;
overwrite_param_names = char('gammap_increment');
overwrite = [gammap_increment];

[zdatalinear2, zdatass2, oobase2_, Mbase2_ ] = ...
    solve_no_constraint(modopt,...
    shockssequence,irfshock,nperiods);


prefix1 = 's1_';
prefix2 = 's2_';
prefix3 = 's3_';
for this_var = 1:Mbase_.endo_nbr
    eval([prefix1,deblank(Mbase_.endo_names{this_var,:}),'_irf=zdatapiecewise(:,this_var);'])
    eval([prefix1,deblank(Mbase_.endo_names{this_var,:}),'_ss=zdatass(this_var);'])
    
    eval([prefix2,deblank(Mbase2_.endo_names{this_var,:}),'_irf=zdatalinear2(:,this_var);'])
    eval([prefix2,deblank(Mbase2_.endo_names{this_var,:}),'_ss=zdatass2(this_var);'])
    
    eval([prefix3,deblank(Mbase3_.endo_names{this_var,:}),'_irf=zdatapiecewise3(:,this_var);'])
    eval([prefix3,deblank(Mbase3_.endo_names{this_var,:}),'_ss=zdatass3(this_var);'])

end


%% Plots

spread_title = 'Expected equity return spread';
spread_title = [spread_title newline '(risky-safe)'];

titlelist = char('Total output', 'Bank capital requirement',...
    'Consumption', 'Credit/GDP ratio',...
    spread_title,  'Total capital');


line1 = 100*[ s1_ytot_irf/s1_ytot_ss, 100*s1_gamma_var_irf,...
    s1_c_irf/s1_c_ss, s1_ltot2ytot_irf/4,...
    [s1_r_e_upper_bar_irf(2:end)+s1_r_e_upper_bar_ss-s1_r_e_irf(2:end)-s1_r_e_ss; nan], s1_ktot_irf/s1_ktot_ss];

line2 = 100*[ s2_ytot_irf/s2_ytot_ss, 100*s2_gamma_var_irf,...
    s2_c_irf/s2_c_ss, s2_ltot2ytot_irf/4,...
    [s2_r_e_upper_bar_irf(2:end)-s2_r_e_irf(2:end); nan], s2_ktot_irf/s2_ktot_ss];

line3 = 100*[ s3_ytot_irf/s3_ytot_ss, 100*s3_gamma_var_irf,...
    s3_c_irf/s3_c_ss, s3_ltot2ytot_irf/4,...
    [s3_r_e_upper_bar_irf(2:end)+s3_r_e_upper_bar_ss-s3_r_e_irf(2:end)-s3_r_e_ss; nan], s3_ktot_irf/s3_ktot_ss];


percent = 'Percent';
ppt = 'Perc. Point';
bpt = 'Basis Point';
level = 'Level';
ylabels = char(percent,bpt,...
    percent,ppt,...
    ppt,percent);
figlabel = '';



legendlist  = char('No-failure Capital Requirements','Calibrated Capital Buffer','No Capital Buffer');
[figure_handle, legend_handle]=makechart(titlelist,legendlist,figlabel,ylabels,line2,line3,line1);
set(legend_handle,'location','NorthEast')
set(legend_handle,'FontSize',8)
printpref;

print -depsc2 figure_4_TFP_shock.eps

% figure for the slides
%reorder = [1 3 2 5 6 4];
%makechart_slide(titlelist(reorder,:),legendlist,figlabel,ylabels(reorder,:),line2(:,reorder),line1(:,reorder));
%printpref


